<?php

    namespace Drupal\configurator_form\Form;

    use Drupal\Core\Ajax\InsertCommand;
    use Drupal\Core\Form\FormBase;
    use Drupal\Core\Form\FormStateInterface;
    use Drupal\Core\Ajax\AjaxResponse;
    use Drupal\Core\Ajax\InvokeCommand;
    use Drupal\Core\Ajax\AppendCommand;
    use Drupal\Core\Ajax\HtmlCommand;
    use Drupal\Core\Url;
    use Drupal\Core\Image\Image;

    class ConfiguratorForm extends FormBase {
        private $categories;
        private $opening_types;
        private $material_types;
        private $timber_types;
        private $window_types;

        public function __construct() {
            $this->categories = $this->getOptions('product_categories');
            $this->opening_types = $this->getRelatedOptions('product_opening_types', key($this->categories));
            $this->material_types = $this->getRelatedOptions('material_types', key($this->opening_types));
            $this->timber_types = $this->getOptions('timber_types');
            $this->window_types = $this->getOptions('window_types');
        }

        public function getFormId(): string {
            // Here we set a unique form id
            return 'configurator_form';
        }

        public function buildForm(array $form, FormStateInterface $form_state, $username = NULL): array {
            $selected_category = $form_state->getValue('category') ?? key($this->categories);

            // Radio buttons form elements.
            $form['category'] = array (
                '#type' => 'radios',
                '#options' => $this->categories,
                '#prefix' => '<div class="configurator__product-categories configurator--radios">',
                '#suffix' => '</div>',
                '#ajax' => [
                    'callback' => array($this, 'populateOpeningTypes'),
                    'wrapper' => 'types',
                    'progress' => array('message' => '')
                ],
                '#attributes' => array('hidden' => true),
                '#theme_wrappers' => array(),
            );

            $form['category_previous'] = array(
                '#type' => 'value',
                '#default_value' => $selected_category,
            );

            if ($form_state->getValue('category') !== $form_state->getValue('category_previous')) {
                $form_state->setValue('opening_types', null);
                $form_state->setValue('material_types', null);
                $form_state->setValue('timber_types', null);
            }

            $form['types'] = array(
                '#type' => 'container',
                '#prefix' => '<div id="types">',
                '#suffix' => '</div>',
            );

            $form['types']['opening_types'] = array(
                '#type' => 'radios',
                '#prefix' => '<div id="opening-types" class="configurator__product-opening-types configurator--radios">',
                '#suffix' => '</div>',
                '#options' => $this->getOpeningTypesOptions($form_state),
                '#ajax' => [
                    'callback' => array($this, 'populateMaterialTypes'),
                    'event' => 'change',
                    'wrapper' => 'material-types',
                    'progress' => array('message' => '')
                ],
                '#attributes' => array('hidden' => true),
            );

            $form['types']['material_types'] = array(
                '#type' => 'radios',
                '#prefix' => '<div id="material-types" class="configurator__material-types configurator--radios">',
                '#suffix' => '</div>',
                '#options' => $this->getMaterialTypesOptions($form_state),
                '#ajax' => [
                    'callback' => array($this, 'populateTimberTypes'),
                    'event' => 'change',
                    'wrapper' => 'timber-types',
                    'progress' => array('message' => '')
                ],
                '#attributes' => array('hidden' => true),
                '#theme_wrappers' => array(),
            );

            $form['types']['timber_types'] = array(
                '#type' => 'radios',
                '#prefix' => '<div id="timber-types" class="configurator__timber-types configurator--radios">',
                '#suffix' => '</div>',
                '#options' => $this->getTimberOptions($form_state),
                '#attributes' => array('hidden' => true),
                '#theme_wrappers' => array(),
            );

            $form['window_types'] = array(
                '#type' => 'radios',
                '#prefix' => '<div id="window-types">',
                '#suffix' => '</div>',
                '#options' => $this->window_types,
                '#ajax' => [
                    'callback' => array($this, 'populateWindowTypes'),
                    'event' => 'change',
                    'wrapper' => 'window-types-container',
                    'progress' => array('message' => '')
                ],
                '#attributes' => array('hidden' => true),
                '#theme_wrappers' => array(),
            );

            $form['window_types']['container'] = array(
                '#type' => 'container',
                '#prefix' => '<div id="window-types-container">',
                '#suffix' => '</div>',
                '#weight' => 1
            );

            $form['window_types']['container']['opening_type'] = array(
                '#type' => 'radios',
                '#prefix' => '<div id="window-types-opening-types">',
                '#suffix' => '</div>',
                '#options' => $this->getRelatedOptions('window_opening_types', $form_state->getValue('window_types')),
            );

            $form['window_types']['container']['options'] = array(
                '#type' => 'radios',
                '#prefix' => '<div id="window-types-options">',
                '#suffix' => '</div>',
                '#options' => $this->getRelatedOptions('window_options', $form_state->getValue('window_types')),
            );

            //submit button.
            $form['actions']['submit'] = array(
                '#type' => 'submit',
                '#value' => $this->t('Save'),
                '#button_type' => 'primary',
            );

            return $form;
        }

        public function validateForm(array &$form, FormStateInterface $form_state): void {
//            if (strlen($form_state->getValue('mobile_number')) < 10) {
//                $form_state->setErrorByName('mobile_number', $this->t('Mobile number too short.'));
//            }
        }

        public function submitForm(array &$form, FormStateInterface $form_state): void {
            \Drupal::messenger()->addMessage($this->t('@can_name ,Your application is being submitted!', array('@can_name' => $form_state->getValue('first_name'))));
            foreach ($form_state->getValues() as $key => $value) {
                \Drupal::messenger()->addMessage($key . ': ' . $value);
            }
            $form_state->setRedirect('<front>');
        }

        function populateOpeningTypes(array $form, FormStateInterface $form_state) {
            return $form['types'];
        }

        function populateMaterialTypes(array $form, FormStateInterface $form_state): array {
            return $form['types']['material_types'];
        }

        function populateTimberTypes(array $form, FormStateInterface $form_state): array {
            return $form['types']['timber_types'];
        }

        function populateWindowTypes(array $form, FormStateInterface $form_state): array {
            return $form['window_types']['container'];
        }

        private function getOptions($name, $parent = 0): array {
            if (!isset($parent)) {
                return [];
            }
            $terms = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->loadTree($name, $parent, null, true);
            $options = [];
            foreach ($terms as $term) {
                if ($term->hasField('field_image') && $term->get('field_image')->entity !== null) {
                    $image = \Drupal::service('image.factory')->get($term->get('field_image')->entity->getFileUri());
                    $file_url = \Drupal::service('file_url_generator')->generateAbsoluteString($image->getSource());
                    $options[$term->id()] = '<img alt="image" src="' . $file_url . '" width="'. $image->getWidth() .'" 
                    height="'. $image->getHeight() .'"/><span>' . $term->getName() . '</span>';
                } else {
                    $options[$term->id()] = $term->getName();
                }
            }
            if (count($options) === 0) {
                return [];
            }

            return $options;
        }

        private function getRelatedOptions($name, $related_term = 0): array {
            if (!isset($related_term)) {
                return [];
            }
            $terms = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->loadTree($name, 0, null, true);

            $options = [];
            foreach ($terms as $term) {
                if (intval($term->get('field_related_taxonomy')[0]->target_id) === intval($related_term)) {
                    if ($term->hasField('field_image') && $term->get('field_image')->entity !== null) {
                        $image = \Drupal::service('image.factory')->get($term->get('field_image')->entity->getFileUri());
                        $file_url = \Drupal::service('file_url_generator')->generateAbsoluteString($image->getSource());
                        $options[$term->id()] = '<img alt="image" src="' . $file_url . '" width="'. $image->getWidth() .'" 
                    height="'. $image->getHeight() .'"/><span>' . $term->getName() . '</span>';
                    } else {
                        $options[$term->id()] = $term->getName();
                    }
                }
            }
            if (count($options) === 0) {
                return [];
            }

            return $options;
        }

        private function getOpeningTypesOptions(FormStateInterface $form_state): array {
            $category = $form_state->getValue('category');
            if ($category !== null) {
                $form_state->set('material_types', null);
                $form_state->set('timber_types', null);
                return $this->getRelatedOptions('product_opening_types', $category);
            }
            return [];
        }

        private function getMaterialTypesOptions(FormStateInterface $form_state): array {
            $opening_type = $form_state->getValue('opening_types');
            if ($opening_type) {
                $form_state->set('timber_types', null);
                return $this->getRelatedOptions('material_types', $opening_type);
            }
            return [];
        }

        private function getTimberOptions(FormStateInterface $form_state): array {
            $material_type = $form_state->getValue('material_types');
            if ($material_type !== null && $material_type !== '46') {
                return $this->getOptions('timber_types');
            }
            return [];
        }
    }
